﻿;*******************************************************************************
;	Title	優先度制御ライブラリ
;	Author	NightWork
;	Ver		20251006
;
;-------------------------------------------------------------------------------
;	プロセスのCPU優先度、I/O優先度、メモリ優先度を操作するライブラリ。
;	自プロセスの他、プロセスID指定により他プロセスの操作が可能。
;
;*******************************************************************************


;*******************************************************************************
; プロセスの優先度設定
;*******************************************************************************
;===============================================================================
;	引数	dwPriority	優先度を指定
;							24:リアルタイム(要管理者権限)
;							13:高
;							10:通常以上
;							 8:通常
;							 6:通常以下
;							 4:低
;
;			dwPid		プロセスIDを指定、省略時は自プロセス。
;
;-------------------------------------------------------------------------------
;	返り値	 0:コマンドは実行された（成否を問わない）
;			-1:指定されたプロセスが存在しない
;			-2:優先度指定が間違ってる
;
;===============================================================================
SetProcessPriority( dwPriority, dwPid:=0) {
	;定数
	PROCESS_QUERY_INFORMATION	:= 0x0400
	PROCESS_SET_INFORMATION		:= 0x0200

	REALTIME_PRIORITY_CLASS		:= 0x00000100
	HIGH_PRIORITY_CLASS			:= 0x00000080
	ABOVE_NORMAL_PRIORITY_CLASS	:= 0x00008000
	NORMAL_PRIORITY_CLASS		:= 0x00000020
	BELOW_NORMAL_PRIORITY_CLASS	:= 0x00004000
	IDLE_PRIORITY_CLASS			:= 0x00000040

	;変数
	hProcess		:= 0	;プロセスハンドル
	dwPriorityClass	:= 0	;優先度クラス

	;---------------------------------------
	;引数型チェック
	If dwPriority is not integer
		Return -2

	;引数内容チェック
	If ( 24	== dwPriority)
		dwPriorityClass	:= REALTIME_PRIORITY_CLASS
	If ( 13	== dwPriority)
		dwPriorityClass	:= HIGH_PRIORITY_CLASS
	If ( 10	== dwPriority)
		dwPriorityClass	:= ABOVE_NORMAL_PRIORITY_CLASS
	If (  8	== dwPriority)
		dwPriorityClass	:= NORMAL_PRIORITY_CLASS
	If (  6	== dwPriority)
		dwPriorityClass	:= BELOW_NORMAL_PRIORITY_CLASS
	If (  4	== dwPriority)
		dwPriorityClass	:= IDLE_PRIORITY_CLASS

	If ( 0 == dwPriorityClass)
		Return -2

	;---------------------------------------
	;プロセスハンドル取得
	If ( 0 != dwPid)
	{
		;別プロセス
		hProcess	:=	DllCall( "OpenProcess"
								, "UInt"	, PROCESS_QUERY_INFORMATION	| PROCESS_SET_INFORMATION
								, "Int"		, False
								, "UInt"	, dwPid)

		If ( 0 == hProcess)
			Return -1
	}Else{
		;プロセスID未指定時は自プロセスハンドルを取得
		hProcess	:=	DllCall( "GetCurrentProcess")
	}

	;---------------------------------------
	;関数コール
	DllCall( "SetPriorityClass"
			, "UInt"	, hProcess						;ProcessHandle
			, "UInt"	, dwPriorityClass)				;ProcessPriorityClass

	;ハンドルクローズ
	DllCall( "CloseHandle"
			, "UInt"	, hProcess)

	Return 0
}



;*******************************************************************************
; プロセスの優先度取得
;*******************************************************************************
;===============================================================================
;	引数	dwPid		プロセスIDを指定、省略時は自プロセス。
;
;-------------------------------------------------------------------------------
;	返り値	優先度
;				24:リアルタイム(要管理者権限)
;				13:高
;				10:通常以上
;				 8:通常
;				 6:通常以下
;				 4:低
;
;				-1:指定されたプロセスが存在しない
;				-2:"GetPriorityClass"の実行に失敗
;
;===============================================================================
GetProcessPriority( dwPid:=0) {
	;定数
	PROCESS_QUERY_INFORMATION	:= 0x0400
	PROCESS_SET_INFORMATION		:= 0x0200

	REALTIME_PRIORITY_CLASS		:= 0x00000100
	HIGH_PRIORITY_CLASS			:= 0x00000080
	ABOVE_NORMAL_PRIORITY_CLASS	:= 0x00008000
	NORMAL_PRIORITY_CLASS		:= 0x00000020
	BELOW_NORMAL_PRIORITY_CLASS	:= 0x00004000
	IDLE_PRIORITY_CLASS			:= 0x00000040

	;変数
	hProcess		:= 0	;プロセスハンドル
	dwPriorityClass	:= 0	;優先度クラス
	dwPriority		:= 0	;優先度

	;---------------------------------------
	;プロセスハンドル取得
	If ( 0 != dwPid)
	{
		;別プロセス
		hProcess	:=	DllCall( "OpenProcess"
								, "UInt"	, PROCESS_QUERY_INFORMATION	| PROCESS_SET_INFORMATION
								, "Int"		, False
								, "UInt"	, dwPid)

		If ( 0 == hProcess)
			Return -1
	}Else{
		;プロセスID未指定時は自プロセスハンドルを取得
		hProcess	:=	DllCall( "GetCurrentProcess")
	}

	;---------------------------------------
	;関数コール
	dwPriorityClass	:=	DllCall( "GetPriorityClass"
								, "UInt"	, hProcess)		;ProcessHandle

	;ハンドルクローズ
	DllCall( "CloseHandle"
			, "UInt"	, hProcess)

	;---------------------------------------
	;取得結果チェック
	If ( 0 == dwPriorityClass)
		Return -2

	If ( REALTIME_PRIORITY_CLASS		== dwPriorityClass)
		dwPriority	:= 24
	If ( HIGH_PRIORITY_CLASS			== dwPriorityClass)
		dwPriority	:= 13
	If ( ABOVE_NORMAL_PRIORITY_CLASS	== dwPriorityClass)
		dwPriority	:= 10
	If ( NORMAL_PRIORITY_CLASS			== dwPriorityClass)
		dwPriority	:=	8
	If ( BELOW_NORMAL_PRIORITY_CLASS	== dwPriorityClass)
		dwPriority	:=	6
	If ( IDLE_PRIORITY_CLASS			== dwPriorityClass)
		dwPriority	:=	4

	Return dwPriority
}



;*******************************************************************************
; プロセスのI/O優先度設定
;*******************************************************************************
;===============================================================================
;	引数	dwIoPriority	I/O優先度を指定
;								2:Normal
;								1:Low
;								0:Very Low
;
;			dwPid		プロセスIDを指定、省略時は自プロセス。
;
;-------------------------------------------------------------------------------
;	返り値	 0:コマンドは実行された（成否を問わない）
;			-1:指定されたプロセスが存在しない
;			-2:優先度指定が間違ってる
;
;===============================================================================
SetProcessIoPriority( dwIoPriority, dwPid:=0) {
	;定数
	PROCESS_INFORMATION_CLASS	:= 0x21			;ProcessIoPriority
	PROCESS_QUERY_INFORMATION	:= 0x0400
	PROCESS_SET_INFORMATION		:= 0x0200

	;変数
	hProcess	:= 0	;プロセスハンドル

	;---------------------------------------
	;引数型チェック
	If dwIoPriority is not integer
		Return -2

	;引数範囲チェック
	If dwIoPriority not between 0 and 2
		Return -2

	;---------------------------------------
	;プロセスハンドル取得
	If ( 0 != dwPid)
	{
		;別プロセス
		hProcess	:=	DllCall( "OpenProcess"
								, "UInt"	, PROCESS_QUERY_INFORMATION	| PROCESS_SET_INFORMATION
								, "Int"		, False
								, "UInt"	, dwPid)

		If ( 0 == hProcess)
			Return -1
	}Else{
		;プロセスID未指定時は自プロセスハンドルを取得
		hProcess	:=	DllCall( "GetCurrentProcess")
	}

	;---------------------------------------
	;関数コール
	DllCall( "Ntdll.dll\NtSetInformationProcess"
			, "UInt"	, hProcess						;ProcessHandle
			, "UInt"	, PROCESS_INFORMATION_CLASS		;ProcessInformationClass
			, "UIntP"	, dwIoPriority					;ProcessInformation
			, "UInt"	, 4)							;ProcessInformationLength

	;ハンドルクローズ
	DllCall( "CloseHandle"
			, "UInt"	, hProcess)

	Return 0
}



;*******************************************************************************
; プロセスのI/O優先度取得
;*******************************************************************************
;===============================================================================
;	引数	dwPid		プロセスIDを指定、省略時は自プロセス。
;
;-------------------------------------------------------------------------------
;	返り値	I/O優先度
;				 2:Normal
;				 1:Low
;				 0:Very Low
;
;				-1:指定されたプロセスが存在しない
;				-2:"NtQueryInformationProcess"の実行に失敗
;
;===============================================================================
GetProcessIoPriority( dwPid:=0) {
	;定数
	PROCESS_INFORMATION_CLASS	:= 0x21			;ProcessIoPriority
	PROCESS_QUERY_INFORMATION	:= 0x0400
	PROCESS_SET_INFORMATION		:= 0x0200

	;変数
	hProcess		:= 0	;プロセスハンドル
	dwIoPriority	:= 0	;優先度
	VarSetCapacity( dwIoPriority, 4)

	;---------------------------------------
	;プロセスハンドル取得
	If ( 0 != dwPid)
	{
		;別プロセス
		hProcess	:=	DllCall( "OpenProcess"
								, "UInt"	, PROCESS_QUERY_INFORMATION	| PROCESS_SET_INFORMATION
								, "Int"		, False
								, "UInt"	, dwPid)

		If ( 0 == hProcess)
			Return -1
	}Else{
		;プロセスID未指定時は自プロセスハンドルを取得
		hProcess	:=	DllCall( "GetCurrentProcess")
	}

	;---------------------------------------
	;関数コール
	ret	:=	DllCall( "Ntdll.dll\NtQueryInformationProcess"
					, "UInt"	, hProcess						;ProcessHandle
					, "UInt"	, PROCESS_INFORMATION_CLASS		;ProcessInformationClass
					, "UIntP"	, dwIoPriority					;ProcessInformation
					, "UInt"	, 4)							;ProcessInformationLength

	;ハンドルクローズ
	DllCall( "CloseHandle"
			, "UInt"	, hProcess)

	If ( ret)
		Return -2

	Return dwIoPriority
}



;*******************************************************************************
; プロセスのメモリ優先度設定
;*******************************************************************************
;===============================================================================
;	引数	dwMemoryPriority	メモリ優先度
;									5:通常		MEMORY_PRIORITY_NORMAL
;									4:通常以下	MEMORY_PRIORITY_BELOW_NORMAL
;									3:中		MEMORY_PRIORITY_MEDIUM
;									2:低		MEMORY_PRIORITY_LOW
;									1:特低		MEMORY_PRIORITY_VERY_LOW
;
;			dwPid		プロセスIDを指定、省略時は自プロセス。
;
;-------------------------------------------------------------------------------
;	返り値	 0:コマンドは実行された（成否を問わない）
;			-1:指定されたプロセスが存在しない
;			-2:優先度指定が間違ってる
;
;===============================================================================
SetProcessMemoryPriority( dwMemoryPriority, dwPid:=0) {
	;定数
	PROCESS_INFORMATION_CLASS	:= 0x00			;ProcessMemoryPriority
	PROCESS_QUERY_INFORMATION	:= 0x0400
	PROCESS_SET_INFORMATION		:= 0x0200

	;変数
	hProcess	:= 0	;プロセスハンドル

	;---------------------------------------
	;引数型チェック
	If dwMemoryPriority is not integer
		Return -2

	;引数範囲チェック
	If dwMemoryPriority not between 1 and 5
		Return -2

	;---------------------------------------
	;プロセスハンドル取得
	If ( 0 != dwPid)
	{
		;別プロセス
		hProcess	:=	DllCall( "OpenProcess"
								, "UInt"	, PROCESS_QUERY_INFORMATION	| PROCESS_SET_INFORMATION
								, "Int"		, False
								, "UInt"	, dwPid)

		If ( 0 == hProcess)
			Return -1
	}Else{
		;プロセスID未指定時は自プロセスハンドルを取得
		hProcess	:=	DllCall( "GetCurrentProcess")
	}

	;---------------------------------------
	;関数コール
	DllCall( "SetProcessInformation"
			, "UInt"	, hProcess						;ProcessHandle
			, "UInt"	, PROCESS_INFORMATION_CLASS		;ProcessInformationClass
			, "UIntP"	, dwMemoryPriority				;ProcessInformation
			, "UInt"	, 4)							;ProcessInformationLength

	;ハンドルクローズ
	DllCall( "CloseHandle"
			, "UInt"	, hProcess)

	Return 0
}



;*******************************************************************************
; プロセスのメモリ優先度取得
;*******************************************************************************
;===============================================================================
;	引数	dwPid		プロセスIDを指定、省略時は自プロセス。
;
;-------------------------------------------------------------------------------
;	返り値	メモリ優先度
;				 5:通常		MEMORY_PRIORITY_NORMAL
;				 4:通常以下	MEMORY_PRIORITY_BELOW_NORMAL
;				 3:中		MEMORY_PRIORITY_MEDIUM
;				 2:低		MEMORY_PRIORITY_LOW
;				 1:特低		MEMORY_PRIORITY_VERY_LOW
;
;				-1:指定されたプロセスが存在しない
;				-2:"GetProcessInformation"の実行に失敗
;
;===============================================================================
GetProcessMemoryPriority( dwPid:=0) {
	;定数
	PROCESS_INFORMATION_CLASS	:= 0x00			;ProcessMemoryPriority
	PROCESS_QUERY_INFORMATION	:= 0x0400
	PROCESS_SET_INFORMATION		:= 0x0200

	;変数
	hProcess			:= 0	;プロセスハンドル
	dwMemoryPriority	:= 0	;優先度
	VarSetCapacity( dwMemoryPriority, 4)

	;---------------------------------------
	;プロセスハンドル取得
	If ( 0 != dwPid)
	{
		;別プロセス
		hProcess	:=	DllCall( "OpenProcess"
								, "UInt"	, PROCESS_QUERY_INFORMATION	| PROCESS_SET_INFORMATION
								, "Int"		, False
								, "UInt"	, dwPid)

		If ( 0 == hProcess)
			Return -1
	}Else{
		;プロセスID未指定時は自プロセスハンドルを取得
		hProcess	:=	DllCall( "GetCurrentProcess")
	}

	;---------------------------------------
	;関数コール
	ret	:=	DllCall( "GetProcessInformation"
					, "UInt"	, hProcess						;ProcessHandle
					, "UInt"	, PROCESS_INFORMATION_CLASS		;ProcessInformationClass
					, "UIntP"	, dwMemoryPriority				;ProcessInformation
					, "UInt"	, 4)							;ProcessInformationLength

	;ハンドルクローズ
	DllCall( "CloseHandle"
			, "UInt"	, hProcess)

	If ( !ret)
		Return -2

	Return dwMemoryPriority
}



;*******************************************************************************
; バックグラウンド処理モードを開始（自プロセスのみ有効）
;*******************************************************************************
;===============================================================================
;	引数	無し
;
;-------------------------------------------------------------------------------
;	返り値	無し
;
;===============================================================================
ProcessModeBackgroundBegin() {
	;定数
	PROCESS_MODE_BACKGROUND_BEGIN	:= 0x00100000

	;関数コール
	DllCall( "SetPriorityClass"
			, "UInt"	, DllCall( "GetCurrentProcess")
			, "UInt"	, PROCESS_MODE_BACKGROUND_BEGIN)

	Return
}



;*******************************************************************************
; バックグラウンド処理モードを終了（自プロセスのみ有効）
;*******************************************************************************
;===============================================================================
;	引数	無し
;
;-------------------------------------------------------------------------------
;	返り値	無し
;
;===============================================================================
ProcessModeBackgroundEnd() {
	;定数
	PROCESS_MODE_BACKGROUND_END	:= 0x00200000

	;関数コール
	DllCall( "SetPriorityClass"
			, "UInt"	, DllCall( "GetCurrentProcess")
			, "UInt"	, PROCESS_MODE_BACKGROUND_END)

	Return
}
